<?php

use Carbon\Carbon;

/**
 * Convert a datetime string into a human-readable "time ago" format.
 *
 * @param string $datetime The datetime string to be converted (e.g., '2024-11-18 12:34:56').
 * @return string A human-readable representation of the time difference (e.g., '2 hours ago').
 */
function humanTime(string $datetime): string
{
    return Carbon::parse($datetime)->diffForHumans();
}

/**
 * Convert a given date to a database-friendly format (e.g., '2024-11-18').
 * If no date is provided, the current date is returned.
 *
 * @param string|null $date The input date string (optional).
 * @return string The formatted date in 'Y-m-d' format.
 */
function dbDate(string|null $date = null): string
{
    return !empty($date)
        ? Carbon::parse($date)->format('Y-m-d')
        : Carbon::now()->format('Y-m-d');
}

/**
 * Convert a given datetime to a database-friendly format (e.g., '2024-11-18 12:34:56').
 * If no datetime is provided, the current datetime is returned.
 *
 * @param string|null $datetime The input datetime string (optional).
 * @return string The formatted datetime in 'Y-m-d H:i:s' format.
 */
function dbDateTime(string|null $datetime = null): string
{
    return !empty($datetime)
        ? Carbon::parse($datetime)->format('Y-m-d H:i:s')
        : Carbon::now()->format('Y-m-d H:i:s');
}

/**
 * Format a given date for frontend display.
 * If no date is provided, the current date is returned.
 * Allows customization of the output format.
 *
 * @param string|null $date The input date string (optional).
 * @param string|null $format The desired date format (optional). Default is 'M d, Y'.
 * @return string The formatted date.
 */
function frontendDate(string|null $date = null, string|null $format = null): string
{
    $format = !empty($format) ? $format : 'M d, Y';

    return !empty($date)
        ? Carbon::parse($date)->format($format)
        : Carbon::now()->format($format);
}

/**
 * Format a given datetime for frontend display.
 * If no datetime is provided, the current datetime is returned.
 * Allows customization of the output format.
 *
 * @param string|null $date The input date string (optional).
 * @param string|null $format The desired date format (optional). Default is 'M d, Y h:i A'.
 * @return string The formatted date.
 */
function frontendDateTime(string|null $datetime = null, string|null $format = null): string
{
    $format = !empty($format) ? $format : 'M d, Y h:i A';

    return !empty($datetime)
        ? Carbon::parse($datetime)->format($format)
        : Carbon::now()->format($format);
}

/**
 * Format the created_at timestamp of a given model for frontend display.
 *
 * If the created_at timestamp is empty, the function returns a string indicating that the model's creation date is not available.
 * Otherwise, it returns the timestamp formatted as a human-readable duration (e.g., "2 hours ago", "3 days ago", etc.)
 *
 * @param \Illuminate\Database\Eloquent\Model $modal The model object.
 * @return string The formatted created_at timestamp.
 */
function createdAt($modal, $isHumanTime = true)
{
    return empty($modal->created_at) ? '<small><i>' . trans('global.text.not_available') . '</i></small>' : isHumanTime($modal->created_at, $isHumanTime);
}

/**
 * Format the updated_at timestamp of a given model for frontend display.
 *
 * If the updated_at timestamp is empty, the function returns a string indicating that the model was never updated.
 * Otherwise, it returns the timestamp formatted as a human-readable duration (e.g., "2 hours ago", "3 days ago", etc.)
 *
 * @param \Illuminate\Database\Eloquent\Model $modal The model object.
 * @return string The formatted updated_at timestamp.
 */
function updatedAt($modal, $isHumanTime = true)
{
    return empty($modal->updated_at) ? '<small><i>' . trans('global.text.never_updated') . '</i></small>' : isHumanTime($modal->created_at, $isHumanTime);
}

/**
 * Check if the given model is soft deleted and format the result for frontend display.
 *
 * If the model is soft deleted (trashed), the function returns an HTML string indicating
 * the model is "Archived". Otherwise, it returns an HTML string indicating the model
 * is "Not Archived".
 *
 * @param \Illuminate\Database\Eloquent\Model $modal The model object to check.
 * @return string The formatted status indicating whether the model is archived or not.
 */
function deletedAt($modal, $isHumanTime = true)
{
    return empty($modal->deleted_at) ? '<small><i>' . trans('global.text.never_updated') . '</i></small>' : isHumanTime($modal->created_at, $isHumanTime);
}

/**
 * Format a given datetime as either a human-readable duration or a formatted date & time,
 * depending on the value of the $isHumanTime parameter.
 *
 * If $isHumanTime is true (default), the function returns a human-readable duration string.
 * Otherwise, it returns a formatted date & time string.
 *
 * @param string $time The datetime string to be converted (e.g., '2024-11-18 12:34:56').
 * @param bool $isHumanTime Whether to return a human-readable duration or a formatted date & time. Default is true.
 * @return string The formatted datetime string.
 */
function isHumanTime($time, $isHumanTime = true)
{
    return $isHumanTime ? humanTime($time) : frontendDateTime($time);
}

/**
 * Returns the first day of the current month, formatted according to the given $format parameter.
 *
 * @param string $format The desired date format (optional). Default is 'm/d/Y'.
 *
 * @return string The formatted first day of the current month.
 */
function getFirstDay($format = 'm/d/Y')
{
    return \Carbon\Carbon::now()->startOfMonth()->format($format);
}

/**
 * Returns the last day of the current month, formatted according to the given $format parameter.
 *
 * @param string $format The desired date format (optional). Default is 'm/d/Y'.
 *
 * @return string The formatted last day of the current month.
 */
function getLastDay($format = 'm/d/Y')
{
    return \Carbon\Carbon::now()->endOfMonth()->format($format);
}
