<?php

namespace App\Http\Controllers\Admin;

use App\Enum\ListingStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\UpdateListingRequest;
use App\Enum\Status;
use App\Models\Listing;
use App\Models\Condition;
use Illuminate\View\View;
use App\Traits\FileUploader;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\VehicleSpecification;
use Illuminate\Http\RedirectResponse;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class ListingController extends Controller implements HasMiddleware
{
    use FileUploader;

    /**
     * Define middleware for the controller.
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:listing.view', only: ['index', 'show', 'edit', 'dataTable']),
            new Middleware('permission:listing.update', only: ['update']),
            new Middleware('permission:listing.approve', only: ['approve', 'reject']),
            new Middleware('permission:listing.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('admin.listings.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid): View
    {
        $listing = Listing::with(['owner', 'specification', 'features', 'condition'])->where('uuid', $uuid)->firstOrFail();

        return view('admin.listings.show', compact('listing'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid): View
    {
        $listing = Listing::with(['owner', 'specification', 'features', 'condition'])->where('uuid', $uuid)->firstOrFail();
        $conditions = Condition::active()->orderBy('sort_order')->get();

        return view('admin.listings.edit', compact('listing', 'conditions'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateListingRequest $request, string $id): RedirectResponse
    {
        $listing = Listing::where('id', $id)->with(['specification', 'features'])->firstOrFail();

        $validated = $request->validated();

        try {
            DB::beginTransaction();

            // Parse uploaded images
            $uploadedImages = json_decode($validated['uploaded_images'], true);
            if (!is_array($uploadedImages) || count($uploadedImages) < 1) {
                return back()->withErrors(['uploaded_images' => 'Please upload at least 1 image.'])->withInput();
            }

            // Update listing
            $listing->update([
                'title' => $validated['title'],
                'description' => $validated['description'] ?? null,
                'price' => $validated['price'],
                'financing_options' => $validated['financing_options'] ?? [],
                'location' => $validated['location'],
                'image_url' => $uploadedImages,
                'status' => ListingStatus::from($validated['status']),
            ]);

            // Update listing condition
            $listing->update([
                'condition_id' => $validated['condition_id'],
            ]);

            // Update vehicle specification
            $listing->specification->update([
                'make' => $validated['make'],
                'model' => $validated['model'],
                'year' => $validated['year'],
                'trim' => $validated['trim'] ?? null,
                'vin' => $validated['vin'],
                'engine_capacity' => $validated['engine_capacity'],
                'fuel_type' => $validated['fuel_type'],
                'transmission' => $validated['transmission'],
                'drive_type' => $validated['drive_type'],
                'mileage' => $validated['mileage'],
                'body_type' => $validated['body_type'],
                'exterior_color' => $validated['exterior_color'],
                'interior_color' => $validated['interior_color'],
                'seats' => $validated['seats'],
                'doors' => $validated['doors'],
            ]);

            // Sync features
            $listing->features()->sync($validated['features'] ?? []);

            DB::commit();

            return redirect()->route('admin.listings.show', $listing->uuid)
                ->with('success', 'Listing updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update listing', [
                'listing_id' => $id,
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return back()->withErrors(['error' => 'Failed to update listing. Please try again.'])->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $uuid): JsonResponse
    {
        try {
            $listing = Listing::where('uuid', $uuid)->first();

            if (empty($listing)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $listing->delete();

            DB::commit();

            return $this->jsonResponse(['message' => 'Listing deleted successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to delete listing', [
                'listing_id' => $id,
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return $this->jsonResponseInternalServerError($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Listings DataTable.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $listings = Listing::with(['owner', 'specification'])->orderBy('created_at', 'desc');

        $dt = DataTables::of($listings);

        // 🔎 Searching
        $dt->filter(function ($query) use ($request) {
            if (!$request->has('search')) return;

            $search = trim($request->input('search')['value']);

            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%$search%")
                    ->orWhere('location', 'like', "%$search%")
                    ->orWhereHas('specification', function ($sq) use ($search) {
                        $sq->where('body_type', 'like', "%$search%")
                            ->orWhere('make', 'like', "%$search%")
                            ->orWhere('model', 'like', "%$search%");
                    })
                    ->orWhereHas('owner', function ($sq) use ($search) {
                        $sq->where('first_name', 'like', "%$search%")
                            ->orWhere('last_name', 'like', "%$search%")
                            ->orWhere('email', 'like', "%$search%");
                    });
            });
        });

        $dt->addColumn('title', fn($record) => $record->title);

        $dt->addColumn('owner', fn($record) => '<div class="user-card">
            <div class="user-info">
                <span class="tb-lead">' . $record->owner->fullName . '</span>
                <span>' . $record->owner->email . '</span>
            </div>
        </div>');

        $dt->addColumn('price', fn($record) => '$' . number_format($record->price, 2));

        $dt->addColumn('location', fn($record) => $record->location);

        $dt->addColumn('type', fn($record) => ucfirst($record->specification?->body_type ?? 'N/A'));

        $dt->addColumn('status', fn($record) => $record->status->badge());

        $dt->addColumn('created', fn($record) => createdAt($record));

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'view', 'syncResponse' => true],
                ['action' => 'update', 'syncResponse' => true],
                ['action' => 'delete'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'admin.listings',
                permissionNamespace: 'listing',
                datatableId: '#listings-dt',
                isLocked: false
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['actions', 'owner', 'status']);

        return $dt->make(true);
    }

    /**
     * Approve a listing.
     */
    public function approve(string $id): JsonResponse
    {
        try {
            $listing = Listing::where('id', $id)->first();

            if (empty($listing)) {
                throw new NotFoundHttpException(trans('global.toast.not_found'));
            }

            DB::beginTransaction();

            $listing->update([
                'status' => ListingStatus::APPROVED,
            ]);

            DB::commit();

            return $this->jsonResponse(['message' => trans('listings.toast.success.approved')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to approve listing', [
                'listing_id' => $id,
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
            ]);
            return $this->jsonResponseInternalServerError(trans('listings.toast.errors.approve_failed'), $e->getCode() ?: 500);
        }
    }

    /**
     * Reject a listing.
     */
    public function reject(string $id): JsonResponse
    {
        try {
            $listing = Listing::where('id', $id)->first();

            if (empty($listing)) {
                throw new NotFoundHttpException(trans('global.toast.not_found'));
            }

            DB::beginTransaction();

            $listing->update([
                'status' => Status::SUSPENDED, // Using SUSPENDED as rejection status
            ]);

            DB::commit();

            return $this->jsonResponse(['message' => trans('listings.toast.success.rejected')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to reject listing', [
                'listing_id' => $id,
                'user_id' => auth()->id(),
                'error' => $e->getMessage(),
            ]);
            return $this->jsonResponseInternalServerError(trans('listings.toast.errors.reject_failed'), $e->getCode() ?: 500);
        }
    }
}
