<?php

namespace App\Http\Controllers\Buyer;

use App\Http\Controllers\Controller;
use App\Repositories\Contracts\ConditionRepositoryInterface;
use App\Repositories\Contracts\ListingRepositoryInterface;
use Illuminate\View\View;
use App\Models\Listing;

class FrontendController extends Controller
{
    public function __construct(
        protected ListingRepositoryInterface $listingRepository,
        protected ConditionRepositoryInterface $conditionRepository
    ) {}

    public function index(): View
    {
        $featuredListings = $this->listingRepository->getFeatured(12);
        $conditions = $this->conditionRepository->getActive();

        // PAGINATED LISTINGS - latest first
        $cars = Listing::with(['specification', 'owner'])
            ->latest()
            ->paginate(9);

        // Transform data for view
        $cars->getCollection()->transform(function ($item) {
            return (object)[
                'name'  => $item->title ?? 'No Title',
                'type'  => $item->specification?->body_type ?? 'Sedan',
                'year'  => $item->specification?->year ?? '----',
                'price' => $item->price ?? 0,
                'image_url' => $item->image_url && is_array($item->image_url) && count($item->image_url) > 0
                    ? $item->image_url[0]
                    : asset('assets/buyer/home/no-image.png'),
                'city'  => $item->location ?? 'Unknown',
                'agent' => $item->owner?->name ?? 'Dealer',
                'added_at' => $item->created_at,
                'specs' => [
                    [
                        'icon' => asset('assets/buyer/home/speedmeter.png'),
                        'value' => $item->specification?->mileage ? $item->specification->mileage . ' Miles' : 'Mileage'
                    ],
                    [
                        'icon' => asset('assets/buyer/home/twinscroll.png'),
                        'value' => $item->specification?->engine_capacity ?? 'Engine'
                    ],
                    [
                        'icon' => asset('assets/buyer/home/gasonline.png'),
                        'value' => $item->specification?->fuel_type ?? 'Fuel'
                    ],
                    [
                        'icon' => asset('assets/buyer/home/automatic.png'),
                        'value' => $item->specification?->transmission ?? 'Auto'
                    ],
                ],
            ];
        });

        return view('buyer.home', [
            'meta_title' => 'Vehicle Listing - Find Your Perfect Car',
            'meta_description' => 'Find your perfect vehicle - Browse our extensive collection of quality vehicles from trusted dealers. Search by make, model, price, and condition.',
            'meta_keywords' => 'vehicles, cars, buy car, sell car, dealers, used cars, new cars, car listings',
            'featuredListings' => $featuredListings,
            'conditions' => $conditions,
            'cars' => $cars,
        ]);
    }
}
