<?php

namespace App\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

abstract class Controller
{
    /**
     * Return JSON/Ajax response for web app
     *
     * @return JsonResponse
     */
    public function jsonResponse($data, $status = 200, $message = 'The given data was invalid.'): JsonResponse
    {
        $status = is_int($status) && $status > 99 ? $status : 500;

        if (is_object($data) && method_exists($data, 'errors'))
            return response()->json(['message' => $message, 'errors' => $data->errors()], $status);

        if (is_array($data) && !($status >= 200 && $status < 300))
            return response()->json(['message' => $message, 'errors' => $data], $status);

        return response()->json($data, $status);
    }

    /**
     * Return JSON/Ajax error response for API
     *
     * @param $errors
     * @param $data
     * @param $status
     * @return JsonResponse
     */
    public function jsonResponseInternalServerError($errors, $data = array(), $status = 500): JsonResponse
    {
        $message = "Internal server error";

        if ($errors instanceof \Exception){
            Log::error($errors->getMessage(), $errors->getTrace());
            if (!app()->isProduction()) $message = $errors->getMessage();
        }elseif (is_string($errors) && !app()->isProduction()){
            $message = $errors;
        }

        return $this->jsonResponse($data, $status, $message);
    }

    /**
     * Return JSON/Ajax validated response for API
     *
     * @param $message
     * @param $data
     * @param $status
     * @return JsonResponse
     */
    public function jsonResponseValidated($message, $data = array(), $status = 422): JsonResponse
    {
        if (is_string($message)) $data = array_merge(['message' => $message], $data);

        return $this->jsonResponse($data, $status, $message);
    }

    /**
     * Return JSON/Ajax not found response for API
     *
     * @param string $message
     * @param array $data
     * @param $status
     * @return JsonResponse
     */
    public function jsonResponseNotFound($message = 'Record not found.', $data = array(), $status = JsonResponse::HTTP_NOT_FOUND)
    {
        if (is_string($message)) $data = array_merge(['message' => $message], $data);

        return $this->jsonResponse($data, $status, $message);
    }

    /**
     * Return JSON/Ajax success response for API
     *
     * @param $message
     * @param $data
     * @param $status
     * @return JsonResponse
     */
    public function jsonResponseSuccess($message = 'Success', $data = array(), $status = JsonResponse::HTTP_OK): JsonResponse
    {
        if (is_string($message)) $data = array_merge(['message' => $message], $data);

        return $this->jsonResponse($data, $status, $message);
    }
}
