<?php

namespace App\Http\Controllers;

use App\Enum\Status;
use App\Models\Role;
use App\Models\User;
use App\Models\DexUser;
use App\Models\Listing;
use App\Models\DxCustomer;
use App\Enum\ListingStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;

class DashboardController extends Controller implements HasMiddleware
{
    /**
     * Define middleware for the controller.
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:dashboard.view', only: ['index']),
        ];
    }

    public function index()
    {
        // Total Admin Users
        $totalAdminUsers = User::role([Role::SUPER_ADMIN])->count();

        // Active Admins
        $activeAdmins = User::role(Role::SUPER_ADMIN)
                            ->where('status', Status::ACTIVE)
                            ->count();

        // Suspended Admins
        $suspendedAdmin = User::role(Role::SUPER_ADMIN)
                            ->where('status', Status::SUSPENDED)
                            ->count();


        // Total Buyers (Customers)
        $buyersCount = User::role(Role::BUYER)->count();

        // Active Buyers
        $activeBuyers = User::role(Role::BUYER)
                            ->where('status', Status::ACTIVE)
                            ->count();

        // Suspended Buyers
        $suspendedBuyers = User::role(Role::BUYER)
                               ->where('status', Status::SUSPENDED)
                               ->count();

        // Total Dealer
        $totalDealer = User::role(Role::DEALER)->count();

        // Active Dealer
        $activeDealer = User::role(Role::DEALER)
                            ->where('status', Status::ACTIVE)
                            ->count();

        // Suspended Dealers
        $suspendedDealer = User::role(Role::DEALER)
                            ->where('status', Status::SUSPENDED)
                            ->count();

        // Total Dex Users
        $totalDexUser = DexUser::count();

        // Total Dx Customers
        $totalDxCustomer = DxCustomer::count();

        // Total Listings
        $totallistings = Listing::count();

        // Active Listings
        $activeListings = Listing::where('status', ListingStatus::ACTIVE->value)->count();

        // Inactive Listings
        $inactiveListings = Listing::where('status', ListingStatus::INACTIVE->value)->count();

        // Pending Listings
        $pendingListings = Listing::where('status', ListingStatus::PENDING->value)->count();

        //  Counts
        $buyersCount = User::role(Role::BUYER)->where('status', Status::ACTIVE)->count();
        $dealersCount = User::role(Role::DEALER)->where('status', Status::ACTIVE)->count();
        $activeListingsCount = \App\Models\Listing::where('status', Status::ACTIVE)->count();
        $pendingListingsCount = \App\Models\Listing::where('status', Status::PENDING)->count();

        // Chart Data: Listings by Status
        $listingsByStatus = \App\Models\Listing::select('status', DB::raw('count(*) as total'))
            ->groupBy('status')
            ->get()
            ->map(function ($item) {
                return [
                    'label' => $item->status->label(),
                    'value' => $item->total,
                    'color' => match ($item->status) {
                        Status::ACTIVE => '#1ee0ac', // Success
                        Status::PENDING => '#09c2de', // Info
                        Status::SUSPENDED => '#e85347', // Danger
                        default => '#854fff', // Primary
                    }
                ];
            });

        // Chart Data: User Registration Trend (Last 30 Days)
        $userTrend = User::select(DB::raw('DATE(created_at) as date'), DB::raw('count(*) as total'))
            ->where('created_at', '>=', now()->subDays(30))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $userTrendLabels = $userTrend->pluck('date');
        $userTrendData = $userTrend->pluck('total');

        // Recent Activity
        $recentListings = \App\Models\Listing::with('owner')->latest()->take(5)->get();
        $recentDealers = User::role(Role::DEALER)->latest()->take(5)->get();

        return view('dashboards.index', compact(
            'totalAdminUsers',
            'activeAdmins',
            'suspendedAdmin',

            'buyersCount',
            'activeBuyers',
            'suspendedBuyers',

            'totalDealer',
            'activeDealer',
            'suspendedDealer',

            'totalDexUser',
            'totalDxCustomer',

            'totallistings',
            'activeListings',
            'inactiveListings',
            'pendingListings',

            'dealersCount',
            'activeListingsCount',
            'pendingListingsCount',
            'listingsByStatus',
            'userTrendLabels',
            'userTrendData',
            'recentListings',
            'recentDealers'

        ));


    }
}
