<?php

namespace App\Http\Controllers\Dealer;

use App\Enum\ListingStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Dealer\StoreListingRequest;
use App\Http\Requests\Dealer\UpdateListingRequest;
use App\Models\Listing;
use App\Models\Condition;
use App\Models\VehicleSpecification;
use App\Traits\FileUploader;
use Yajra\DataTables\Facades\DataTables;
use App\Services\DataTableActionLinksService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\View\View;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class ListingController extends Controller implements HasMiddleware
{
    use FileUploader;

    public static function middleware(): array
    {
        return [
            new Middleware('permission:listing.view', only: ['index', 'show', 'edit', 'dataTable']),
            new Middleware('permission:listing.update', only: ['update']),
            new Middleware('permission:listing.add', only: ['create', 'store']),
            new Middleware('permission:listing.delete', only: ['destroy']),
        ];
    }

    public function index(): View
    {
        return view('dealer.listings.index');
    }

    public function create(): View
    {
        $conditions = Condition::active()->orderBy('sort_order')->get();
        return view('dealer.listings.create', compact('conditions'));
    }

    public function store(StoreListingRequest $request): RedirectResponse
    {
        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $uploadedImages = json_decode($validated['uploaded_images'], true);
            if (!is_array($uploadedImages) || count($uploadedImages) < 3) {
                return back()->withErrors(['uploaded_images' => 'Please upload at least 3 images.'])->withInput();
            }

            $listing = Listing::create([
                'id' => Str::random(20),
                'uuid' => Str::uuid(),
                'title' => $validated['title'],
                'description' => $validated['description'] ?? null,
                'price' => $validated['price'],
                'financing_options' => $validated['financing_options'] ?? [],
                'location' => $validated['location'],
                'image_url' => $uploadedImages,
                'owner_id' => auth()->id(),
                'status' => ListingStatus::PENDING,
                'condition_id' => $validated['condition_id'],
            ]);

            VehicleSpecification::create([
                'listing_id' => $listing->id,
                'make' => $validated['make'],
                'model' => $validated['model'],
                'year' => $validated['year'],
                'trim' => $validated['trim'] ?? null,
                'vin' => $validated['vin'],
                'engine_capacity' => $validated['engine_capacity'],
                'fuel_type' => $validated['fuel_type'],
                'transmission' => $validated['transmission'],
                'drive_type' => $validated['drive_type'],
                'mileage' => $validated['mileage'],
                'body_type' => $validated['body_type'],
                'exterior_color' => $validated['exterior_color'],
                'interior_color' => $validated['interior_color'],
                'seats' => $validated['seats'],
                'doors' => $validated['doors'],
            ]);

            if (!empty($validated['features'])) {
                $listing->features()->attach($validated['features']);
            }

            DB::commit();

            return redirect()->route('dealers.listings.index')
                ->with('success', trans('listings.toast.success.created'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Dealer listing create failed', [
                'error' => $e->getMessage(),
            ]);
            return back()->withErrors(['error' => trans('listings.toast.errors.create_failed')])->withInput();
        }
    }

    /**
     * ✅ SAME AS ADMIN
     */
    public function show(string $uuid): View
    {
        $listing = Listing::with(['owner', 'specification', 'features', 'condition'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        return view('dealer.listings.show', compact('listing'));
    }

    /**
     * ✅ SAME AS ADMIN
     */
    public function edit(string $uuid): View
    {
        $listing = Listing::with(['owner', 'specification', 'features', 'condition'])
            ->where('uuid', $uuid)
            ->firstOrFail();

        $conditions = Condition::active()->orderBy('sort_order')->get();

        return view('dealer.listings.edit', compact('listing', 'conditions'));
    }

    /**
     * ✅ FIXED: UUID instead of ID
     */
    public function update(UpdateListingRequest $request, string $uuid): RedirectResponse
    {
        $listing = Listing::where('uuid', $uuid)
            ->with(['specification', 'features'])
            ->firstOrFail();

        $validated = $request->validated();

        try {
            DB::beginTransaction();

            $uploadedImages = json_decode($validated['uploaded_images'], true);
            if (!is_array($uploadedImages) || count($uploadedImages) < 1) {
                return back()->withErrors(['uploaded_images' => 'Please upload at least 1 image.'])->withInput();
            }

            $listing->update([
                'title' => $validated['title'],
                'description' => $validated['description'] ?? null,
                'price' => $validated['price'],
                'financing_options' => $validated['financing_options'] ?? [],
                'location' => $validated['location'],
                'image_url' => $uploadedImages,
                'condition_id' => $validated['condition_id'],
            ]);

            $listing->specification->update([
                'make' => $validated['make'],
                'model' => $validated['model'],
                'year' => $validated['year'],
                'trim' => $validated['trim'] ?? null,
                'vin' => $validated['vin'],
                'engine_capacity' => $validated['engine_capacity'],
                'fuel_type' => $validated['fuel_type'],
                'transmission' => $validated['transmission'],
                'drive_type' => $validated['drive_type'],
                'mileage' => $validated['mileage'],
                'body_type' => $validated['body_type'],
                'exterior_color' => $validated['exterior_color'],
                'interior_color' => $validated['interior_color'],
                'seats' => $validated['seats'],
                'doors' => $validated['doors'],
            ]);

            $listing->features()->sync($validated['features'] ?? []);

            DB::commit();

            return redirect()->route('dealers.listings.index')
                ->with('success', trans('listings.toast.success.updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Dealer listing update failed', [
                'error' => $e->getMessage(),
            ]);
            return back()->withErrors(['error' => trans('listings.toast.errors.update_failed')])->withInput();
        }
    }

    /**
     * ✅ SAME AS ADMIN
     */
    public function destroy(string $uuid): JsonResponse
    {
        try {
            $listing = Listing::where('uuid', $uuid)->firstOrFail();

            DB::beginTransaction();
            $listing->delete();
            DB::commit();

            return response()->json(['message' => trans('listings.toast.success.deleted')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Dealer listing delete failed', [
                'error' => $e->getMessage(),
            ]);
            return response()->json(['message' => trans('listings.toast.errors.delete_failed')], 500);
        }
    }

     public function dataTable(Request $request): JsonResponse
    {
        $listings = Listing::with(['owner', 'specification'])->orderBy('created_at', 'desc');

        $dt = DataTables::of($listings);

        // 🔎 Searching
        $dt->filter(function ($query) use ($request) {
            if (!$request->has('search')) return;

            $search = trim($request->input('search')['value']);

            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%$search%")
                    ->orWhere('location', 'like', "%$search%")
                    ->orWhereHas('specification', function ($sq) use ($search) {
                        $sq->where('body_type', 'like', "%$search%")
                            ->orWhere('make', 'like', "%$search%")
                            ->orWhere('model', 'like', "%$search%");
                    })
                    ->orWhereHas('owner', function ($sq) use ($search) {
                        $sq->where('first_name', 'like', "%$search%")
                            ->orWhere('last_name', 'like', "%$search%")
                            ->orWhere('email', 'like', "%$search%");
                    });
            });
        });

        $dt->addColumn('title', fn($record) => $record->title);

        $dt->addColumn('owner', fn($record) => '<div class="user-card">
            <div class="user-info">
                <span class="tb-lead">' . $record->owner->fullName . '</span>
                <span>' . $record->owner->email . '</span>
            </div>
        </div>');

        $dt->addColumn('price', fn($record) => '$' . number_format($record->price, 2));

        $dt->addColumn('location', fn($record) => $record->location);

        $dt->addColumn('type', fn($record) => ucfirst($record->specification?->body_type ?? 'N/A'));

        $dt->addColumn('status', fn($record) => $record->status->badge());

        $dt->addColumn('created', fn($record) => createdAt($record));

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'view', 'syncResponse' => true],
                ['action' => 'update', 'syncResponse' => true],
                ['action' => 'delete'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'dealers.listings',
                permissionNamespace: 'listing',
                datatableId: '#listings-dt',
                isLocked: false
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['actions', 'owner', 'status']);

        return $dt->make(true);
    }
}
