<?php

namespace App\Http\Controllers\Dealers;

use App\Http\Controllers\Controller;
use App\Models\Dealers\DealerBranches;
use App\Models\State;
use App\Services\DataTableActionLinksService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Yajra\DataTables\DataTables;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class BranchController extends Controller implements HasMiddleware
{
    /**
     * Define middleware for this controller.
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:dealers.branch.view', only: ['index', 'dataTable', 'show']),
            new Middleware('permission:dealers.branch.add', only: ['create', 'store']),
            new Middleware('permission:dealers.branch.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of branches.
     */
    public function index(): View
    {
        return view('branches.index');
    }

    /**
     * Show the form for creating a new branch.
     */
    public function create(): View
    {

        return view('branches.modals.form');
    }

    /**
     * Store a newly created branch.
     */
    public function store(Request $request): JsonResponse
    {
        $request->validate([
            'branch_name' => 'required|string|max:255',
            'address'     => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            DealerBranches::create([
                'dealer_id'   => auth()->user()->model_id,
                'branch_name' => $request->input('branch_name'),
                'address'     => $request->input('address'),
            ]);

            DB::commit();

            return $this->jsonResponse('Branch created successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Display a specific branch.
     */
    public function show(string $id): View|JsonResponse
    {
        try {
            $branch = DealerBranches::whereUuid($id)->first();

            if (empty($branch)) {
                throw new NotFoundHttpException('Branch not found.');
            }

            return view('branches.modals.show', compact('branch'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Show the form for editing a branch.
     */
    public function edit(string $id): View|JsonResponse
    {
        try {
            $branch = DealerBranches::whereUuid($id)->first();

            if (empty($branch)) {
                throw new NotFoundHttpException('Branch not found.');
            }

            $states = State::all();

            return view('branches.modals.form', compact('branch', 'states'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Update the specified branch.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'branch_name' => 'required|string|max:255',
            'address'     => 'required|string|max:255',
        ]);

        try {
            $branch = DealerBranches::whereUuid($id)->first();

            if (empty($branch)) {
                throw new NotFoundHttpException('Branch not found.');
            }

            DB::beginTransaction();

            $branch->update([
                'branch_name' => $request->input('branch_name'),
                'address'     => $request->input('address'),
            ]);

            DB::commit();

            return $this->jsonResponse('Branch updated successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Remove the specified branch.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $branch = DealerBranches::whereUuid($id)->first();

            if (empty($branch)) {
                throw new NotFoundHttpException('Branch not found.');
            }

            $branch->delete();

            return $this->jsonResponse('Branch deleted successfully.');
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * DataTable endpoint for branches.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $model = DealerBranches::where('dealer_id', auth()->user()->model_id)->latest();

        $dt = DataTables::of($model);

        $dt->filter(function ($query) use ($request) {
            if (empty($request->input('search')['value'] ?? null)) {
                return;
            }

            $search = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($q) use ($keywords) {
                foreach ($keywords as $word) {
                    $q->where('branch_name', 'like', "%$word%")
                      ->orWhere('address', 'like', "%$word%");
                }
            });
        });

        $dt->addColumn('branch_name', function ($record) {
            return ucwords($record->branch_name);
        });

        $dt->addColumn('address', function ($record) {
            return canEmpty($record->address);
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'update', 'modalSize' => 'lg'],
                ['action' => 'view', 'modalSize' => 'lg'],
                ['action' => 'delete'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'branches',
                permissionNamespace: 'dealers.branch',
                datatableId: '#branches-dt',
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['branch_name', 'address', 'created', 'updated', 'actions']);

        return $dt->make(true);
    }
}
