<?php

namespace App\Http\Controllers\Documentations;

use Illuminate\Support\Str;
use App\Models\Documentation;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use App\Services\ActivityLogs\DocumentationLogService;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use App\Http\Requests\Documentation\DocumentationStoreRequest;
use App\Http\Requests\Documentation\DocumentationUpdateRequest;

class DocumentationController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:documentation.view', only: ['index', 'dataTable', 'show']),
            new Middleware('permission:documentation.add', only: ['create', 'store']),
            new Middleware('permission:documentation.update', only: ['edit', 'update']),
            new Middleware('permission:documentation.delete', only: ['destroy']),
            new Middleware('permission:documentation.restore', only: ['restore']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view('documentations.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('documentations.modals.edit');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(DocumentationStoreRequest $request)
    {
        try {
            DB::beginTransaction();

            Documentation::create([
                'key' => 'custom',
                'title' => $request->input('title'),
                'slug' => Str::slug($request->input('title')),
                'content' => $request->input('content'),
                'updated_at' => null,
            ]);

            (new DocumentationLogService())->created();

            DB::commit();

            return $this->jsonResponse(trans('documentations.toast.success.created'), JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $slug)
    {
        $documentation = Documentation::whereSlug($slug)->firstOrFail();

        $routesList = [
            'user.view' => 'users',
            'role.view' => 'roles',
            'terms.view' => 'terms_conditions',
            'activity_log.view' => 'activity_logs',
        ];

        $key = $documentation->key;

        // Check if the key matches any value in routesList
        $requiredPermission = array_search($key, $routesList);

        if ($requiredPermission !== false) {
            // If the key matches and the user has the required permission
            if (auth()->user() && auth()->user()->can($requiredPermission)) {
                return view('documentations.show', compact('documentation'));
            } else {
                abort(403);
            }
        } else {
            // If the key does not match any value in routesList, allow access
            return view('documentations.show', compact('documentation'));
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        $documentation = Documentation::whereUuid($uuid)->first();
        if (empty($documentation)) return $this->jsonResponse(trans('global.toast.not_found'), JsonResponse::HTTP_NOT_FOUND);

        return view('documentations.modals.edit', compact('documentation'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(DocumentationUpdateRequest $request, string $uuid)
    {
        try {
            $documentation = Documentation::whereUuid($uuid)->first();

            if (empty($documentation)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $documentation->update([
                'title' => $request->input('title'),
                'slug' => Str::slug($request->input('title')),
                'content' => $request->input('content'),
            ]);

            $documentation->refresh();

            $newTitle = $documentation->title;

            (new DocumentationLogService())->updated($documentation);

            DB::commit();

            return $this->jsonResponse(trans('documentations.toast.success.updated'), JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $uuid)
    {
        try {
            $documentation = Documentation::deletable()->whereUuid($uuid)->first();
            if (empty($documentation)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $documentation->delete();

            (new DocumentationLogService())->deleted($documentation);

            DB::commit();

            return $this->jsonResponse(trans('documentations.toast.success.deleted'), JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    public function restore(string $id): JsonResponse
    {
        try {
            $documentation = Documentation::withTrashed()->whereUuid($id)->first();
            if (empty($documentation)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $documentation->restore();

            (new DocumentationLogService())->restored($documentation);

            DB::commit();

            return $this->jsonResponse(['message' => trans('documentations.toast.success.restored')]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    public function dataTable()
    {
        $documentations = Documentation::orderBy('id');

        $documentations = $documentations->get();

        $routesList = [
            'user.view' => 'users',
            'role.view' => 'roles',
            'terms.view' => 'terms_conditions',
            'activity_log.view' => 'activity_logs',
            'ticket.view' => 'tickets',
        ];

        // Filter the documentations to include only those with keys that match values in routesList and the user has the appropriate permissions
        $filteredDocumentations = $documentations->filter(function ($documentation) use ($routesList) {
            $requiredPermission = array_search($documentation->key, $routesList);
            return $requiredPermission === false || (auth()->user() && auth()->user()->can($requiredPermission));
        });

        $dt = DataTables::of($filteredDocumentations);

        $dt->addColumn('key', function ($record) {
            return ucwords(str_replace('_', ' ', $record->key));
        });

        $dt->addColumn('title', function ($record) {
            return !empty($record->title)
                ? '<a href="' . route('documentations.show', $record->slug) . '">' . $record->title . '</a>'
                : canEmpty($record->title);
        });

        $dt->addColumn('content', function ($record) {
            return canEmpty(addEllipsis(strip_tags($record->content)));
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('action', function ($record) {
            $links = [
                ['action' => 'update', 'modalSize' => 'xl'],
                ['action' => 'delete', 'shouldRender' => $record->is_deletable],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'documentations',
                permissionNamespace: 'documentation',
                datatableId: '#documentations-dt'
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['key', 'title', 'content', 'created', 'updated', 'action']);

        return $dt->make(true);
    }

    public function archivedDataTable()
    {
        $documentations = Documentation::orderBy('id')->onlyTrashed();

        $documentations = $documentations->get();

        $routesList = [
            'user.view' => 'users',
            'role.view' => 'roles',
            'terms.view' => 'terms_conditions',
            'activity_log.view' => 'activity_logs',
            'ticket.view' => 'tickets',
        ];

        // Filter the documentations to include only those with keys that match values in routesList and the user has the appropriate permissions
        $filteredDocumentations = $documentations->filter(function ($documentation) use ($routesList) {
            $requiredPermission = array_search($documentation->key, $routesList);
            return $requiredPermission === false || (auth()->user() && auth()->user()->can($requiredPermission));
        });

        $dt = DataTables::of($filteredDocumentations);

        $dt->addColumn('key', function ($record) {
            return ucwords(str_replace('_', ' ', $record->key));
        });

        $dt->addColumn('title', function ($record) {
            return !empty($record->title)
                ? '<a href="' . route('documentations.show', $record->slug) . '">' . $record->title . '</a>'
                : canEmpty($record->title);
        });

        $dt->addColumn('content', function ($record) {
            return canEmpty(addEllipsis(strip_tags($record->content)));
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('action', function ($record) {
            $links = [
                ['action' => 'restore'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'documentations',
                permissionNamespace: 'documentation',
                datatableId: '#archived-documentations-dt'
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['key', 'title', 'content', 'created', 'updated', 'action']);

        return $dt->make(true);
    }
}
