<?php

namespace App\Http\Controllers\DxCustomer;

use App\Enum\Status;
use App\Models\DxCustomer;
use App\Models\State;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class DxCustomerController extends Controller implements HasMiddleware
{
    /**
     * Get the middleware for the controller.
     *
     * @return array
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:dx-customer.view', only: ['index', 'dataTable', 'show']),
            new Middleware('permission:dx-customer.add', only: ['create', 'store']),
            new Middleware('permission:dx-customer.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('dx-customers.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {

        return view('dx-customers.modals.form');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'customer_name' => 'required|string|max:255',
        ]);

        try {
            DB::beginTransaction();

            dxCustomer::create([
                'customer_name' => $request->input('customer_name'),
                'status' => Status::ACTIVE,
        ]);

            DB::commit();

            return $this->jsonResponse('Customer created successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $customer = dxCustomer::whereUuid($id)->first();

            if (empty($customer)) throw new \Exception('Customer not found.');

            return view('dx-customers.modals.show', compact('customer'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        try {
            $customer = DxCustomer::whereUuid($id)->first();

            if (empty($customer)) throw new \Exception('Customer not found.');

            return view('dx-customers.modals.form', compact('customer'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $request->validate([
           'customer_name' => 'required|string|max:255',
        ]);

        try {
            $customer = DxCustomer::whereUuid($id)->first();

            if (empty($customer)) throw new \Exception('Customer not found.');

            DB::beginTransaction();

            $customer->update([
                'customer_name' => $request->input('customer_name'),
                'updated_at' => now(),
            ]);

            DB::commit();

            return $this->jsonResponse('Customer updated successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $customer = DxCustomer::whereUuid($id)->first();

            if (empty($customer)) throw new NotFoundHttpException('Customer not found.');

            $customer->delete();

            return $this->jsonResponse('Customer deleted successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

        /**
     * Activate a specific customer.
     */
    public function activate(string $id): JsonResponse
    {
        try {
            $customer = DxCustomer::whereUuid($id)->first();

            if (empty($customer)) {
                throw new NotFoundHttpException('Customer not found.');
            }

            $customer->update(['status' => Status::ACTIVE]);

            return $this->jsonResponse('Customer activated successfully.');
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Deactivate a specific customer.
     */
    public function deactivate(string $id): JsonResponse
    {
        try {
            $customer = DxCustomer::whereUuid($id)->first();

            if (empty($customer)) {
                throw new NotFoundHttpException('Customer not found.');
            }

            $customer->update(['status' => Status::INACTIVE]);

            return $this->jsonResponse('Customer deactivated successfully.');
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Return the listing of the resource.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $model = DxCustomer::latest();

        $dt = DataTables::of($model);

        $dt->filter(function ($query) use ($request) {
            if (empty($request->input('search'))) return;

            $search = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($query) use ($keywords) {
                foreach ($keywords as $word) {
                    $query->where(function($query) use ($word){
                        $query->where('customer_name', 'like', "%$word%");
                    });
                }
            });
        });

        $dt->addColumn('customer_name', function ($record) {
            return ucwords($record->customer_name);
        });

        $dt->addColumn('status', function ($record) {
            return $record->getStatusBadge();
        });


        $dt->addColumn('created', function ($record) {
            return humanTime($record->created_at);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'update', 'modalSize' => 'lg'],
                [
                    'action' => 'custom',
                    'shouldRender' => auth()->user()->can('dx-customer.activate') && $record->status == Status::INACTIVE,
                    'url' => route('dx-customers.activate', $record->uuid),
                    'attributes' => 'confirm-btn data-method="post" data-datatable="#dx-customer-dt"
                        data-title="' . trans('dx-customer.trans.confirmable.activate.title') . '"
                        data-message="' . trans('dx-customer.trans.confirmable.activate.message') . '"
                        data-confirm-btn-text="' . trans('dx-customer.trans.confirmable.activate.button') . '"
                        data-cancel-btn-text="' . trans('dx-customer.trans.confirmable.button.cancel') . '"',
                    'icon' => 'check',
                    'buttonText' => trans('dx-customer.trans.table.actions.activate'),
                    'syncResponse' => true,
                ],
                [
                    'action' => 'custom',
                    'shouldRender' => auth()->user()->can('dx-customer.deactivate') && $record->status == Status::ACTIVE,
                    'url' => route('dx-customers.deactivate', $record->uuid),
                    'attributes' => 'confirm-btn data-method="post" data-datatable="#dx-customer-dt"
                        data-title="' . trans('dx-customer.trans.confirmable.deactivate.title') . '"
                        data-message="' . trans('dx-customer.trans.confirmable.deactivate.message') . '"
                        data-confirm-btn-text="' . trans('dx-customer.trans.confirmable.deactivate.button') . '"
                        data-cancel-btn-text="' . trans('dx-customer.trans.confirmable.button.cancel') . '"',
                    'icon' => 'check',
                    'buttonText' => trans('dx-customer.trans.table.actions.deactivate'),
                    'syncResponse' => true,
                ],
                ['action' => 'view', 'modalSize' => 'lg'],
                ['action' => 'delete']
            ];
            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'dx-customers',
                permissionNamespace: 'dx-customer',
                datatableId: '#dx-customer-dt',
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['customer_name', 'status', 'created', 'updated', 'actions']);

        return $dt->make(true);
    }
}
