<?php

namespace App\Http\Controllers\Pages;

use App\Models\Page;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class PageController extends Controller implements HasMiddleware
{
    /**
     * Get the middleware for the controller.
     *
     * @return array
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:page.view', only: ['index', 'dataTable', 'show']),
            new Middleware('permission:page.add', only: ['create', 'store']),
            new Middleware('permission:page.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('pages.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        return view('pages.modals.edit');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'description' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // Generate a unique slug
            $slug = generateUniqueSlug(new Page, $request->input('name'));

            Page::create([
                'name' => $request->input('name'),
                'slug' => $slug,
                'description' => $request->input('description'),
            ]);

            DB::commit();

            return $this->jsonResponse('Page created successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $page = Page::whereUuid($id)->first();

            if (empty($page)) throw new \Exception('Page not found.');

            return view('pages.modals.show', compact('page'));
        } catch (\Exception $exception) {
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        try {
            $page = Page::whereUuid($id)->first();

            if (empty($page)) throw new \Exception('Page not found.');

            return view('pages.modals.edit', compact('page'));
        } catch (\Exception $exception) {
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id): JsonResponse
    {
        $request->validate([
            'name' => 'required|string',
            'description' => 'nullable|string',
        ]);

        try {
            $page = Page::whereUuid($id)->first();

            if (empty($page)) throw new \Exception('Page not found.');

            DB::beginTransaction();

            $page->update([
                'name' => $request->input('name'),
                'description' => $request->input('description'),
            ]);

            DB::commit();

            return $this->jsonResponse('Page updated successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $page = Page::whereUuid($id)->first();

            if (empty($page)) throw new NotFoundHttpException('Page not found.');

            if (!$page->isDeletable()) throw new \Exception('Page is not deletable.');

            DB::beginTransaction();

            $page->delete();

            DB::commit();

            return $this->jsonResponse('Page deleted successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Return the listing of the resource.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $categories = Page::latest();

        $dt = DataTables::of($categories);

        $dt->filter(function ($query) use ($request) {
            if (empty($request->input('search'))) return;

            $search = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($query) use ($keywords) {
                foreach ($keywords as $word) {
                    $query->where('name', 'like', "%$word%")
                        ->orWhere('description', 'like', "%$word%");
                }
            });
        });

        $dt->addColumn('name', function ($record) {
            return ucwords($record->name);
        });

        $dt->addColumn('description', function ($record) {
            return canEmpty(addEllipsis(strip_tags($record->description), 80));
        });

        $dt->addColumn('created', function ($record) {
            return $record->createdAt();
        });

        $dt->addColumn('updated', function ($record) {
            return $record->updatedAt();
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                [
                    'action' => 'update',
                    'modalSize' => 'lg'
                ],
                ['action' => 'view', 'modalSize' => 'lg'],
                ['action' => 'delete']
            ];
            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'pages',
                permissionNamespace: 'page',
                datatableId: '#pages-dt',
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['name', 'description', 'created', 'updated', 'actions']);

        return $dt->make(true);
    }
}
