<?php

namespace App\Http\Controllers\Profile;

use App\Models\Term;
use Illuminate\View\View;
use App\Traits\FileUploader;
use Illuminate\Http\Request;
use App\Models\Dealers\Dealer;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\Profile\ProfileUpdateRequest;

class ProfileController extends Controller
{
    use FileUploader;

    public function index(): View
    {
        $user = auth()->user();

        if ($user->isDealer()) {
            $dealer = Dealer::with(['user', 'representatives', 'finances', 'addresses', 'branches', 'operations'])
                ->whereHas('user', function ($q) use ($user) {
                    $q->where('id', $user->id);
                })
                ->first();

            if (!$dealer) {
                abort(404, 'Dealer profile not found.');
            }

            return view('profile.dealers.index', compact('dealer'));
        }

        return view('profile.index');
    }

    /**
     * Display the user's profile form.
     */
    public function edit(): View
    {
        $user = auth()->user();

        if ($user->isDealer()) {
            $dealer = Dealer::with(['user', 'representatives', 'finances', 'addresses', 'branches', 'operations', 'media'])
                ->whereHas('user', function ($q) use ($user) {
                    $q->where('id', $user->id);
                })
                ->first();

            if (!$dealer) {
                abort(404, 'Dealer profile not found.');
            }

            return view('profile.dealers.modals.profile', compact('dealer'));
        }

        return view('profile.modals.profile');
    }

    /**
     * Update the user's profile information.
     */
    public function update(ProfileUpdateRequest $request): JsonResponse
    {
        try {
            $only = [
                'first_name',
                'last_name',
                'gender',
                'dob',
            ];

            auth()->user()->update($request->only($only));

            $params = [];

            foreach ($only as $field) {
                $params[$field] = auth()->user()?->$field;
            }

            $params['name'] = auth()->user()->fullName;
            $params['dob'] = frontendDate(auth()->user()->dob);

            return $this->jsonResponse([
                'message' => 'Profile updated successfully.',
                'params' => $params
            ], JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Delete the user's account.
     */
    public function destroy(Request $request)
    {
        $request->validateWithBag('userDeletion', [
            'password' => ['required', 'current_password'],
        ]);

        $user = $request->user();

        auth()->logout();

        $user->delete();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->to('/');
    }

    /**
     * Show the terms of services page for the authenticated user.
     *
     * @return \Illuminate\Contracts\View\View
     */
    public function terms(): View
    {
        $term = Term::wherePosition('terms_of_services')->first();

        return view('profile.terms', compact('term'));
    }

    public function editCompany(): View
    {
        $dealer = Dealer::with(['user', 'addresses'])
            ->whereHas('user', fn($q) => $q->where('id', auth()->user()->id))
            ->first();

        return view('profile.dealers.modals.company-info', compact('dealer'));
    }

    public function updateCompany(Request $request): JsonResponse
    {
        $dealer = Dealer::with('addresses')
            ->whereHas('user', fn($q) => $q->where('id', auth()->user()->id))
            ->first();

        try {
            DB::beginTransaction();

            // Update dealer core info
            $dealer->update($request->only([
                'company_name',
                'registration_number',
                'tax_identification_number',
            ]));

            // Refresh address (delete old and create new)
            $dealer->addresses()->delete();
            $dealer->addresses()->create([
                'country'     => $request->input('country'),
                'state'       => $request->input('state'),
                'city'        => $request->input('city'),
                'street'      => $request->input('street'),
                'postal_code' => $request->input('postal_code'),
            ]);

            DB::commit();
            return $this->jsonResponse(['message' => 'Company information updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    public function editRepresentative(): View
    {
        $dealer = Dealer::with(['user', 'representatives'])
            ->whereHas('user', function ($q) {
                $q->where('id', auth()->user()->id);
            })
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }
        return view('profile.dealers.modals.representative-info', compact('dealer'));
    }

    public function updateRepresentative(Request $request): JsonResponse
    {
        $dealer = Dealer::with('representatives')
            ->whereHas('user', fn($q) => $q->where('id', auth()->user()->id))
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        try {
            DB::beginTransaction();

            // Remove old reps (simple overwrite strategy)
            $dealer->representatives()->delete();

            $dealer->representatives()->create([
                'full_name'     => $request->input('rep_name'),
                'nic_number'    => $request->input('rep_cnic'),
                'position'      => $request->input('rep_position'),
                'contact_phone' => $request->input('rep_contact'),
            ]);

            DB::commit();
            return $this->jsonResponse(['message' => 'Representative information updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    public function editFinancial(): View
    {
        $dealer =  Dealer::with(['user', 'finances'])
            ->whereHas('user', function ($q) {
                $q->where('id', auth()->user()->id);
            })
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        return view('profile.dealers.modals.financial-info', compact('dealer'));
    }

    public function updateFinancial(Request $request): JsonResponse
    {
        $dealer = Dealer::with('finances')
            ->whereHas('user', fn($q) => $q->where('id', auth()->user()->id))
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        try {
            DB::beginTransaction();

            // Replace old finance records
            $dealer->finances()->delete();

            $dealer->finances()->create([
                'name'                => $request->input('bank_name'),
                'account_holder_name' => $request->input('account_holder_name'),
                'account_number'      => $request->input('account_number'),
            ]);

            DB::commit();
            return $this->jsonResponse(['message' => 'Financial information updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse(['message' => $e->getMessage()], 500);
        }
    }

    public function editBusiness(): View
    {
        $dealer =  Dealer::with(['user', 'branches', 'operations'])
            ->whereHas('user', function ($q) {
                $q->where('id', auth()->user()->id);
            })
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        return view('profile.dealers.modals.business-info', compact('dealer'));
    }

    public function updateBusiness(Request $request): JsonResponse
    {
        $dealer = Dealer::with(['branches', 'operations'])
            ->whereHas('user', fn($q) => $q->where('id', auth()->id()))
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        try {
            DB::beginTransaction();

            // Reset branches & operations
            $dealer->branches()->delete();
            $dealer->operations()->delete();

            // Recreate branches
            if ($request->filled('branches')) {
                foreach ($request->input('branches') as $branch) {
                    $dealer->branches()->create([
                        'branch_name' => $branch['branch_name'] ?? null,
                        'status'      => $branch['status'] ?? 1,
                    ]);
                }
            }

            // Recreate operations
            if ($request->filled('operations')) {
                foreach ($request->input('operations') as $operation) {
                    $dealer->operations()->create([
                        'operation' => $operation['operation'],
                    ]);
                }
            }

            DB::commit();
            return $this->jsonResponse(['message' => 'Business information updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse(['message' => $e->getMessage()], 500);
        }
    }


    public function editDocuments(): View
    {
        $dealer = Dealer::with(['user', 'media'])
            ->whereHas('user', function ($q) {
                $q->where('id', auth()->user()->id);
            })
            ->first();

        if (!$dealer) {
            abort(404, 'Dealer profile not found.');
        }

        return view('profile.dealers.modals.documents-info', compact('dealer'));
    }

    public function updateDocuments(Request $request): JsonResponse
    {
        $dealer = Dealer::with('media')
            ->whereHas('user', fn($q) => $q->where('id', auth()->id()))
            ->first();

        try {
            DB::beginTransaction();

            $fileFields = [
                'bank_swift',
                'business_license',
                'tax_certificate',
                'bank_statement',
                'company_certificate',
            ];

            foreach ($fileFields as $field) {
                if ($request->hasFile($field)) {
                    $file = $request->file($field);
                    $filePath = $this->uploadPublicFile($file, 'dealers/documents');

                    $old = $dealer->media()->where('type', $field)->first();
                    if ($old) {
                        $this->removePublicFile($old->file_path);
                        $old->delete();
                    }

                    // Save new file in media
                    $dealer->media()->create([
                        'type'          => $field,
                        'file_path'     => $filePath,
                        'original_name' => $file->getClientOriginalName(),
                        'mime_type'     => $file->getClientMimeType(),
                        'size'          => $file->getSize(),
                    ]);
                }
            }

            DB::commit();
            return $this->jsonResponse(['message' => 'Documents updated successfully.'], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse(['message' => $e->getMessage()], 500);
        }
    }
}
