<?php

namespace App\Http\Controllers\Settings;

use App\Models\SystemSetting;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use App\Http\Requests\Admin\Settings\MailUpdateRequest;

class MailController extends Controller implements HasMiddleware
{
    private $allowedKeys = [
        'mail_host',
        'mail_port',
        'mail_username',
        'mail_password',
        'mail_encryption',
        'mail_from_email',
        'mail_from_name',
    ];

    public static function middleware(): array
    {
        return [
            new Middleware('permission:mail_setting.view', only: ['index']),
            new Middleware('permission:mail_setting.update', only: ['edit', 'update']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        try {
            $mailSettings = SystemSetting::whereIn('key', $this->allowedKeys)->get()->pluck('value', 'key');

            return view('settings.mail.index', compact('mailSettings'));
        } catch (\Exception $e) {
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit()
    {
        try {
            $mailSettings = SystemSetting::whereIn('key', $this->allowedKeys)->get()->pluck('value', 'key');

            return view('settings.mail.modals.edit', compact('mailSettings'));
        } catch (\Exception $e) {
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Show the form for update the specified resource.
     */
    public function update(MailUpdateRequest $request)
    {
        try {
            DB::beginTransaction();

            foreach ($this->allowedKeys as $key) {
                if (!$request->has($key)) continue;

                SystemSetting::updateOrCreate([
                    'key' => $key,
                ], [
                    'value' => $request->input($key) ?? false,
                    'updated_at' => now(),
                ]);
            }

            DB::commit();

            // Clear the env settings cache to ensure the new settings are loaded
            Artisan::call('optimize:clear');

            $params = [];

            foreach ($this->allowedKeys as $key) {
                $params[$key] = $request->input($key);
            }

            return $this->jsonResponse([
                'message' => trans('system_settings.mail_settings.toast.success.updated'),
                'params' => $params
            ], JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }
}
