<?php

namespace App\Http\Controllers\Taxes;

use App\Models\Tax;
use App\Models\State;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class TaxController extends Controller implements HasMiddleware
{
    /**
     * Get the middleware for the controller.
     *
     * @return array
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:tax.view', only: ['index', 'dataTable', 'show']),
            new Middleware('permission:tax.add', only: ['create', 'store', 'edit']),
            new Middleware('permission:tax.delete', only: ['destroy']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('taxes.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        $states = State::whereHas('country', function ($query) {
            $query->where('iso', 'us');
        })->get();

        return view('taxes.modals.form', compact('states'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'state_id'     => 'required|integer|exists:states,id',
            'tax'          => 'required|numeric|min:0',
            'custom_tax'   => 'required|numeric|min:0',
            'shipping_fee' => 'required|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            Tax::create([
                'state_id'     => $request->input('state_id'),
                'tax'          => $request->input('tax'),
                'custom_tax'   => $request->input('custom_tax'),
                'shipping_fee' => $request->input('shipping_fee'),
            ]);

            DB::commit();

            return $this->jsonResponse(__('tax.messages.create_success'));
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        try {
            $tax = Tax::whereUuid($uuid)->firstOrFail();

            return view('taxes.modals.show', compact('tax'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        try {
            $tax = Tax::whereUuid($uuid)->firstOrFail();

            $states = State::whereHas('country', function ($query) {
                $query->where('iso', 'us');
            })->get();

            return view('taxes.modals.form', compact('tax', 'states'));
        } catch (\Exception $exception) {
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $uuid): JsonResponse
    {
        $request->validate([
            'state_id'     => 'required|integer|exists:states,id',
            'tax'          => 'required|numeric|min:0',
            'custom_tax'   => 'required|numeric|min:0',
            'shipping_fee' => 'required|numeric|min:0',
        ]);

        try {
            $tax = Tax::whereUuid($uuid)->firstOrFail();

            DB::beginTransaction();

            $tax->update([
                'state_id'     => $request->input('state_id'),
                'tax'          => $request->input('tax'),
                'custom_tax'   => $request->input('custom_tax'),
                'shipping_fee' => $request->input('shipping_fee'),
            ]);

            DB::commit();

            return $this->jsonResponse(__('tax.messages.update_success'));
        } catch (\Exception $exception) {
            DB::rollBack();
            Log::error($exception->getMessage(), $exception->getTrace());
            return $this->jsonResponseInternalServerError($exception);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $uuid): JsonResponse
    {
        try {
            $tax = Tax::whereUuid($uuid)->with(['state'])->firstOrFail();

            $tax->delete();

            return $this->jsonResponse(__('tax.messages.delete_success'));
        } catch (\Exception $exception) {
            DB::rollBack();
            return $this->jsonResponseInternalServerError($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Return the listing of the resource for DataTable.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $model = Tax::latest();
        $dt    = DataTables::of($model);

        $dt->filter(function ($query) use ($request) {
            if (empty($request->input('search'))) {
                return;
            }

            $search   = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($query) use ($keywords) {
                foreach ($keywords as $word) {
                    $query->whereHas('state', function ($query) use ($word) {
                        $query->where('name', 'like', "%$word%");
                    });
                }
            });
        });

        $dt->addColumn('name', function ($record) {
            return ucwords($record->state->name);
        });

        $dt->addColumn('tax', function ($record) {
            return canEmpty($record->tax . '%');
        });

        $dt->addColumn('custom_tax', function ($record) {
            return canEmpty($record->custom_tax . '%');
        });

        $dt->addColumn('shipping_fee', function ($record) {
            return canEmpty($record->shipping_fee);
        });

        $dt->addColumn('created', function ($record) {
            return humanTime($record->created_at);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'update', 'modalSize' => 'lg'],
                ['action' => 'view', 'modalSize' => 'lg'],
                ['action' => 'delete'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'taxes',
                permissionNamespace: 'tax',
                datatableId: '#taxes-dt',
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['uuid', 'name', 'tax', 'custom_tax', 'shipping_fee', 'created', 'updated', 'actions']);

        return $dt->make(true);
    }
}
