<?php

namespace App\Http\Controllers\Terms;

use Exception;
use App\Models\Term;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Services\DataTableActionLinksService;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use App\Services\ActivityLogs\TermsAndConditionsLogService;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

class TermController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('permission:term.view', only: ['index', 'dataTable', 'page']),
            new Middleware('permission:term.update', only: ['edit', 'update']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return view('terms.index');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $uuid)
    {
        try {
            $term = Term::whereUuid($uuid)->first();

            if (empty($term)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            return view('terms.modals.show', compact('term'));
        } catch (\Exception $exception) {
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $uuid)
    {
        try {
            $term = Term::whereUuid($uuid)->first();
            if (empty($term)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            return view('terms.modals.edit', compact('term'));
        } catch (\Exception $exception) {
            return $this->jsonResponse($exception->getMessage(), $exception->getCode());
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $uuid)
    {
        try {
            $term = Term::whereUuid($uuid)->first();
            if (empty($term)) throw new NotFoundHttpException(trans('global.toast.not_found'));

            DB::beginTransaction();

            if (
                empty($request->input('title')) &&
                ($term->position == 'footer' || in_array($term->position, ['guest_footer_terms_condition', 'guest_footer_privacy_policy']))
            ) {
                throw new BadRequestException(trans('global.toast.unable_to_update'));
            }

            $term->update([
                'title' => in_array($term->position, ['login_page', 'api_key', 'terms_of_services']) ? null : $request->input('title'),
                'slug' => Str::slug($request->input('title')),
                'content' => $request->input('content'),
            ]);

            (new TermsAndConditionsLogService())->updated($term);

            DB::commit();

            return $this->jsonResponse(trans('terms_conditions.toast.success.updated'), JsonResponse::HTTP_OK);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Show the listing of the resource
     */
    public function dataTable(Request $request)
    {
        $terms = Term::query();

        $dt = DataTables::of($terms);

        $dt->filter(function ($query) use ($request) {
            if ($request->has('search')) {
                $search = "%{$request->input('search')['value']}%";

                $query->where('position', 'like', $search)
                    ->orWhere('title', 'like', $search)
                    ->orWhere('content', 'like', $search);
            }
        });

        $dt->addColumn('position', function ($record) {
            return ucwords(str_replace('_', ' ', $record->position));
        });

        $dt->addColumn('title', function ($record) {
            return canEmpty($record->title);
        });

        $dt->addColumn('content', function ($record) {
            return addEllipsis($record->content);
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('action', function ($record) {
            $links = [
                ['action' => 'update', 'modalSize' => 'xl'],
                ['action' => 'view'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'terms',
                permissionNamespace: 'term',
                datatableId: '#terms-dt'
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['position', 'title', 'content', 'created', 'updated', 'action']);

        return $dt->make(true);
    }
}
