<?php

namespace App\Http\Controllers\Users;

use App\Enum\Status;
use App\Models\Role;
use App\Models\User;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\Users\UserStoreRequest;
use App\Services\ActivityLogs\AuthLogService;
use App\Services\ActivityLogs\UserLogService;
use App\Services\DataTableActionLinksService;
use App\Http\Requests\Users\UserUpdateRequest;
use Illuminate\Routing\Controllers\Middleware;
use Illuminate\Routing\Controllers\HasMiddleware;
use Symfony\Component\HttpFoundation\Exception\BadRequestException;
use Symfony\Component\HttpKernel\Exception\UnprocessableEntityHttpException;

class UserController extends Controller implements HasMiddleware
{
    /**
     * Define middleware for the controller.
     */
    public static function middleware(): array
    {
        return [
            new Middleware('permission:user.add', only: ['create', 'store', 'resendInvitation']),
            new Middleware('permission:user.view', only: ['index', 'show', 'comments', 'dataTable']),
            new Middleware('permission:user.update', only: ['edit', 'update']),
            new Middleware('permission:user.delete', only: ['destroy']),
            new Middleware('permission:user.restore', only: ['restore']),
        ];
    }

    /**
     * Display a listing of the resource.
     */
    public function index(): View
    {
        return view('users.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): View
    {
        return view('users.modals.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(UserStoreRequest $request)
    {
        try {
            $invite = Str::random(16);

            DB::beginTransaction();

            $user = User::create([
                'user_id' => auth()->user()->id,
                'first_name' => $request->input('first_name'),
                'last_name' => $request->input('last_name'),
                'email' => $request->input('email'),
                'password' => getUuid(32),
                'status' => Status::INVITED,
                'invite' => $invite,
                'updated_at' => null,
            ]);

            $user->assignRole(Role::USER);

            $user->sendInvitationEmail($invite, Role::USER);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.created')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage(), $e->getTrace());
            return $this->sendServerErrorResponse($e);
        }
    }

    /**
     * Resend invitation link in case of previously undelivered link
     */
    public function resendInvitation(string $id)
    {
        try {
            $user = User::whereUuid($id)->first();
            if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

            if (!$user->isInvited()) throw new BadRequestException(trans('users.toast.errors.unable_resend_invitation'));

            $invite = Str::random(16);

            DB::beginTransaction();

            $user->update(['invite' => $invite]);

            $user->sendInvitationEmail($invite, Role::SUPER_ADMIN);

            (new UserLogService())->reinvited($user);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.invited')]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Approve the account of the resource.
     */
    public function approve($id): JsonResponse
    {
        try {
            $user = User::whereUuid($id)->statusVerified()->first();
            if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $user->update(['status' => Status::ACTIVE]);

            (new UserLogService())->approved($user);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.approved')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage(), $e->getTrace());
            return $this->sendServerErrorResponse($e);
        }
    }

    /**
     * Manually mark the user's email as verified
     */
    public function verifyEmail($id): JsonResponse
    {
        try {
            $user = User::whereUuid($id)->whereNull('email_verified_at')->first();
            if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

            DB::beginTransaction();

            $user->update([
                'status' => Status::VERIFIED,
                'email_verified_at' => now(),
            ]);

            (new AuthLogService())->emailVerified($user);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.verified')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage(), $e->getTrace());
            return $this->sendServerErrorResponse($e);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): View
    {
        $user = User::whereUuid($id)->first();

        if (empty($user)) throw new ModelNotFoundException(trans('global.toast.errors.not_found'));
        return view('users.modals.show', compact('user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id): View
    {
        $user = User::whereUuid($id)->first();
        if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

        if (!in_array($user->status, [Status::ACTIVE, Status::SUSPENDED])) {
            throw new BadRequestException(trans('global.toast.unable_to_update'));
        }

        if ($user->isSuperAdmin() || $user->id === auth()->user()->id)
            throw new BadRequestException(trans('global.toast.unable_to_update'));

        return view('users.modals.edit', compact('user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UserUpdateRequest $request, string $id): JsonResponse
    {
        try {
            $user = User::whereUuid($id)->first();
            if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

            if (!in_array($user->status, [Status::ACTIVE, Status::SUSPENDED]) || $user->isSuperAdmin() || $user->id == auth()->user()->id) {
                throw new BadRequestException(trans('global.toast.unable_to_update'));
            }

            $user->update([
                'first_name' => $request->input('first_name'),
                'last_name' =>  $request->input('last_name'),
                'phone' =>      $request->input('phone'),
                'status' => empty($request->input('status')) ? Status::SUSPENDED : Status::ACTIVE,
                'admin_comments' => empty($request->input('status')) ? $request->input('comments') : null
            ]);

            return $this->jsonResponse(['message' => trans('users.toast.success.updated')]);
        } catch (\Exception $e) {
            DB::rollBack();
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $user = User::whereUuid($id)->first();
            if (empty($user)) throw new ModelNotFoundException(trans('global.toast.not_found'));

            if ($user->id === auth()->user()->id) throw new BadRequestException(trans('global.toast.unable_to_update'));

            if ($user->hasRole('super_admin')) throw new UnprocessableEntityHttpException(trans('global.toast.unable_to_remove'));

            DB::beginTransaction();

            $user->delete();

            (new UserLogService())->deleted($user);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.deleted')]);
        } catch (\Exception $e) {
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Recover the specified resource from storage.
     */
    public function restore($uuid): JsonResponse
    {
        try {
            $user = User::whereUuid($uuid)->onlyTrashed()->firstOrFail();

            if ($user->id === auth()->user()->id) throw new BadRequestException(trans('global.toast.unable_to_update'));

            DB::beginTransaction();

            $user->restore();

            (new UserLogService())->restored($user);

            DB::commit();

            return $this->jsonResponse(['message' => trans('users.toast.success.restored')]);
        } catch (ModelNotFoundException $e) {
            return $this->jsonResponse(['message' => trans('users.toast.errors.not_found')]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error($e->getMessage(), $e->getTrace());
            return $this->jsonResponse($e->getMessage(), $e->getCode());
        }
    }

    /**
     * Show admin comments for the resource.
     */
    public function comments(string $uuid): View
    {
        $user = User::where('uuid', $uuid)->first();
        return view('users.modals.comments', compact('user'));
    }

    /**
     * Return the listing of the resource.
     */
    public function dataTable(Request $request): JsonResponse
    {
        $users = User::role([Role::ADMIN, Role::SUPER_ADMIN])
        ->when(!auth()->user()->isSuperAdmin(), function ($q) {
            return $q->whereUserId(auth()->user()->id);
        })
        ->whereNull('model_type')
        ->orderBy('id', 'desc');

        $dt = DataTables::of($users);

        $dt->filter(function ($query) use ($request) {
            if (!$request->has('search')) return;

            $search = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($query) use ($keywords) {
                foreach ($keywords as $word) {
                    $query->where(function ($query) use ($word) {
                        $query->where('first_name', 'like', "%$word%")
                            ->orWhere('last_name', 'like', "%$word%")
                            ->orWhere('email', 'like', "%$word%")
                            ->orWhere('phone', 'like', "%$word%")
                            ->orWhere('status', 'like', "%$word%");
                    });
                }
            });
        });

        $dt->addColumn('name', function ($record) {
            return '<div class="user-card">
                        <div class="user-avatar ' . getRandomColorClass() . ' d-none d-sm-flex">
                            ' . getAvatarHtml($record) . '
                        </div>
                        <div class="user-info">
                            <span class="tb-lead">' . $record->fullName . '</span>
                            <span>' . $record->email . '</span>
                        </div>
                    </div>';
        });

        $dt->addColumn('phone', function ($record) {
            return canEmpty($record->phone);
        });

        $dt->addColumn('comments', function ($record) {
            return empty($record->admin_comments)
                ? canEmpty(null)
                : '<a href="' . route('users.comments', $record->uuid) . '" class="btn btn-icon btn-sm btn-light" async-modal
                        data-bs-toggle="tooltip" data-bs-placement="bottom" title="' . trans('users.toggle.view_comments') . '" data-method="post">
                        <em class="icon ni ni-comments"></em>
                   </a>';
        });

        $dt->addColumn('status', function ($record) {
            return $record->getStatusBadge();
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'update', 'shouldRender' => auth()->user()->can('user.update') && in_array($record->status, [Status::ACTIVE, Status::SUSPENDED])],
                [
                    'action' => 'custom',
                    'shouldRender' => $record->isInvited() && auth()->user()->can('user.invite'),
                    'url' => route('users.resend-invitation', $record->uuid),
                    'attributes' => 'confirm-btn data-method="post" data-datatable="#users-dt"
                        data-title="' . trans('users.confirmable.resend_invitation.title') . '"
                        data-message="' . trans('users.confirmable.resend_invitation.message') . '"
                        data-confirm-btn-text="' . trans('global.confirmable.button.delete') . '"
                        data-cancel-btn-text="' . trans('global.confirmable.button.cancel') . '"',
                    'icon' => 'reload',
                    'buttonText' => trans('users.table.actions.resend_invitation'),
                    'syncResponse' => true,
                ],
                [
                    'action' => 'custom',
                    'shouldRender' => auth()->user()->can('user.verify_email') && !$record->isVerified(),
                    'url' => route('users.verify-email', $record->uuid),
                    'attributes' => 'confirm-btn data-title="' . trans('users.confirmable.verify.title') . '"
                        data-message="' . trans('users.confirmable.verify.message') . '"
                        data-confirm-btn-text="' . trans('users.confirmable.verify.button') . '"
                        data-cancel-btn-text="' . trans('global.confirmable.button.cancel') . '"
                        data-method="post" data-datatable="#users-dt"',
                    'icon' => 'mail',
                    'buttonText' => trans('users.table.actions.verify_email'),
                    'syncResponse' => true,
                ],
                [
                    'action' => 'custom',
                    'shouldRender' => auth()->user()->can('user.approve') && $record->status == Status::VERIFIED,
                    'url' => route('users.approve', $record->uuid),
                    'attributes' => 'confirm-btn data-method="post" data-datatable="#users-dt"
                        data-title="' . trans('users.confirmable.approve.title') . '"
                        data-message="' . trans('users.confirmable.approve.message') . '"
                        data-confirm-btn-text="' . trans('users.confirmable.approve.button') . '"
                        data-cancel-btn-text="' . trans('global.confirmable.button.cancel') . '"',
                    'icon' => 'check',
                    'buttonText' => trans('users.table.actions.approve_account'),
                    'syncResponse' => true,
                ],
                ['action' => 'view', 'modalSize' => 'lg', 'shouldRender' => auth()->user()->can('user.view')],
                ['action' => 'delete', 'shouldRender' => auth()->user()->can('user.delete') && $record->id !== auth()->user()->id],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'users',
                permissionNamespace: 'user',
                datatableId: '#users-dt',
                isLocked: $record->id == auth()->user()->id
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['actions', 'name', 'phone', 'comments', 'status', 'created', 'updated']);

        return $dt->make(true);
    }

    /**
     * Return the listing of the resource.
     */
    public function archivedDataTable(Request $request): JsonResponse
    {
        $users = User::role([Role::ADMIN, Role::SUPER_ADMIN])
        ->when(!auth()->user()->isSuperAdmin(), function ($q) {
            return $q->whereUserId(auth()->user()->id);
        })
        ->whereNull('model_type')
        ->orderBy('id', 'desc');

        $dt = DataTables::of($users);

        $dt->filter(function ($query) use ($request) {
            if (!$request->has('search')) return;

            $search = trim($request->input('search')['value']);
            $keywords = explode(' ', $search);

            $query->where(function ($query) use ($keywords) {
                foreach ($keywords as $word) {
                    $query->where(function ($query) use ($word) {
                        $query->where('first_name', 'like', "%$word%")
                            ->orWhere('last_name', 'like', "%$word%")
                            ->orWhere('email', 'like', "%$word%")
                            ->orWhere('phone', 'like', "%$word%")
                            ->orWhere('status', 'like', "%$word%");
                    });
                }
            });
        });

        $dt->addColumn('name', function ($record) {
            return '<div class="user-card">
                        <div class="user-avatar ' . getRandomColorClass() . ' d-none d-sm-flex">
                            ' . getAvatarHtml($record) . '
                        </div>
                        <div class="user-info">
                            <span class="tb-lead">' . $record->fullName . '</span>
                            <span>' . $record->email . '</span>
                        </div>
                    </div>';
        });

        $dt->addColumn('phone', function ($record) {
            return canEmpty($record->phone);
        });

        $dt->addColumn('comments', function ($record) {
            return empty($record->admin_comments)
                ? canEmpty(null)
                : '<a href="' . route('users.comments', $record->uuid) . '" class="btn btn-icon btn-sm btn-light" async-modal
                        data-bs-toggle="tooltip" title="View Comments" data-method="post">
                        <em class="icon ni ni-comments"></em>
                   </a>';
        });

        $dt->addColumn('status', function ($record) {
            return $record->getStatusBadge();
        });

        $dt->addColumn('created', function ($record) {
            return createdAt($record);
        });

        $dt->addColumn('updated', function ($record) {
            return updatedAt($record);
        });

        $dt->addColumn('actions', function ($record) {
            $links = [
                ['action' => 'restore'],
            ];

            return (new DataTableActionLinksService(
                model: $record,
                routeNamespace: 'users',
                permissionNamespace: 'user',
                datatableId: '#users-dt',
                isLocked: $record->id == auth()->user()->id
            ))->byArray($links);
        });

        $dt->addIndexColumn();

        $dt->rawColumns(['actions', 'name', 'phone', 'comments', 'status', 'created', 'updated']);

        return $dt->make(true);
    }
}
