<?php

namespace App\Http\Requests\Dealer;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateListingRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $listingId = $this->route('listing'); // This is the listing ID from the route
        
        return [
            // Basic Information
            'make' => 'required|string|max:255',
            'model' => 'required|string|max:255',
            'year' => 'required|integer|min:1900|max:' . (date('Y') + 1),
            'trim' => 'nullable|string|max:255',
            'vin' => [
                'required',
                'string',
                'size:17',
                function ($attribute, $value, $fail) use ($listingId) {
                    $exists = \App\Models\VehicleSpecification::where('vin', $value)
                        ->where('listing_id', '!=', $listingId)
                        ->exists();
                    if ($exists) {
                        $fail('This VIN code has already been used.');
                    }
                },
            ],
            
            // Technical Specifications
            'engine_capacity' => 'required|integer|min:1',
            'fuel_type' => 'required|in:gasoline,diesel,hybrid,electric',
            'transmission' => 'required|in:automatic,manual',
            'drive_type' => 'required|in:AWD,FWD,RWD,4WD',
            'mileage' => 'required|integer|min:0',
            
            // Body and Exterior
            'body_type' => 'required|in:SUV,sedan,pickup,coupe,hatchback,van,convertible',
            'exterior_color' => 'required|string|max:255',
            'interior_color' => 'required|string|max:255',
            'seats' => 'required|integer|min:2|max:20',
            'doors' => 'required|integer|min:2|max:6',
            'condition_id' => 'required|exists:conditions,id',
            
            // Price and Financing
            'price' => 'required|numeric|min:0',
            'financing_options' => 'nullable|array',
            'financing_options.*' => 'in:loan,lease,cash',
            
            // Additional Information
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'location' => 'required|string|max:255',
            
            // Features
            'features' => 'nullable|array',
            'features.*' => 'exists:vehicle_features,id',
            
            // Images (for update, we allow existing images, so minimum is 1)
            'uploaded_images' => 'required|string',
        ];
    }

    /**
     * Get custom messages for validator errors.
     *
     * @return array<string, string>
     */
    public function messages(): array
    {
        return [
            'vin.required' => 'VIN code is required.',
            'vin.size' => 'VIN code must be exactly 17 characters.',
            'vin.unique' => 'This VIN code has already been used.',
            'uploaded_images.required' => 'Please upload at least 1 image of the vehicle.',
            'condition_id.required' => trans('listings.form.condition.label') . ' is required.',
            'condition_id.exists' => 'Selected condition is invalid.',
        ];
    }
}

