<?php

namespace App\Http\Traits;

use Illuminate\Http\JsonResponse;
use Throwable;

trait ApiResponse
{
    /**
     * Core JSON Response builder
     *
     * @param mixed $data
     * @param int $status
     * @param string|null $message
     * @param bool|null $success
     * @return JsonResponse
     */
    public function jsonResponse($data, int $status = 200, ?string $message = null, ?bool $success = null)
    {
        $status = is_int($status) && $status > 99 ? $status : 500;

        // Default success flag if not explicitly passed
        $success = $success ?? ($status >= 200 && $status < 300);

        // Handle Validation Errors (MessageBag / Validator)
        if (is_object($data) && method_exists($data, 'errors')) {
            return response()->json([
                'success' => false,
                'message' => $message ?? 'The given data was invalid.',
                'errors' => $data->errors(),
            ], $status);
        }

        // Handle array errors
        if (is_array($data) && !$success) {
            return response()->json([
                'success' => false,
                'message' => $message ?? 'The given data was invalid.',
                'errors' => $data,
            ], $status);
        }

        // Normal success/error
        return response()->json([
            'success' => $success,
            'message' => $message,
            'data' => $success ? $data : null,
            'errors' => $success ? null : $data,
        ], $status);
    }

    /**
     * Success Response
     */
    public function sendSuccessResponse(string $message, array $data = [], int $status = 200): JsonResponse
    {
        return $this->jsonResponse($data, $status, $message, true);
    }

    /**
     * Validation Error Response
     */
    public function sendValidationErrorResponse($errors, string $message = 'The given data was invalid.', int $status = 422): JsonResponse
    {
        return $this->jsonResponse($errors, $status, $message, false);
    }

    /**
     * Server Error Response
     */
    public function sendServerErrorResponse(Throwable $e, int $status = 500): JsonResponse
    {
        if (app()->environment('production')) {
            return $this->jsonResponse([], $status, 'Server error', false);
        }

        return $this->jsonResponse([], $status, $e->getMessage(), false);

    }
}
