<?php

namespace App\Repositories\Contracts;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

interface ListingRepositoryInterface extends RepositoryInterface
{
    /**
     * Get featured listings.
     *
     * @param int $limit
     * @param array $with
     * @return Collection
     */
    public function getFeatured(int $limit = 12, array $with = ['specification', 'condition', 'owner']): Collection;

    /**
     * Get approved listings with pagination.
     *
     * @param int $perPage
     * @param array $with
     * @param array $filters
     * @return LengthAwarePaginator
     */
    public function getApprovedPaginated(
        int $perPage = 15,
        array $with = ['specification', 'condition', 'owner'],
        array $filters = []
    ): LengthAwarePaginator;

    /**
     * Get listings by owner (dealer).
     *
     * @param int $ownerId
     * @param array $with
     * @param array $filters
     * @return LengthAwarePaginator
     */
    public function getByOwner(
        int $ownerId,
        array $with = ['specification', 'condition'],
        array $filters = []
    ): LengthAwarePaginator;

    /**
     * Get related listings.
     *
     * @param string $listingId
     * @param int $limit
     * @param array $with
     * @return Collection
     */
    public function getRelated(string $listingId, int $limit = 4, array $with = ['specification', 'condition', 'owner']): Collection;

    /**
     * Get listing by UUID with relations.
     *
     * @param string $uuid
     * @param array $with
     * @return Model|null
     */
    public function findByUuidWith(string $uuid, array $with = ['specification', 'condition', 'owner', 'features']): ?Model;

    /**
     * Get listings by condition.
     *
     * @param int $conditionId
     * @param int $perPage
     * @param array $with
     * @return LengthAwarePaginator
     */
    public function getByCondition(
        int $conditionId,
        int $perPage = 15,
        array $with = ['specification', 'condition', 'owner']
    ): LengthAwarePaginator;

    /**
     * Get listing statistics.
     *
     * @param int|null $ownerId
     * @return array
     */
    public function getStatistics(?int $ownerId = null): array;
}

