<?php

namespace App\Repositories\Eloquent;

use App\Models\Dealers\Dealer;
use App\Repositories\Contracts\DealerRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class DealerRepository extends BaseRepository implements DealerRepositoryInterface
{

    /**
     * Get dealer by user ID.
     *
     * @param int $userId
     * @param array $with
     * @return Model|null
     */
    public function findByUserId(int $userId, array $with = ['user']): ?Model
    {
        return $this->model
            ->with($with)
            ->whereHas('user', function ($query) use ($userId) {
                $query->where('id', $userId);
            })
            ->first();
    }

    /**
     * Get verified dealers.
     *
     * @param array $with
     * @return Collection
     */
    public function getVerified(array $with = ['user']): Collection
    {
        return $this->model
            ->with($with)
            ->whereHas('user', function ($query) {
                $query->where('status', \App\Enum\Status::ACTIVE);
            })
            ->get();
    }

    /**
     * Get dealer with full details.
     *
     * @param string $uuid
     * @param array $with
     * @return Model|null
     */
    public function findByUuidWith(
        string $uuid,
        array $with = ['user', 'representatives', 'finances', 'branches', 'operations']
    ): ?Model {
        return $this->model
            ->with($with)
            ->where('uuid', $uuid)
            ->first();
    }
}

