<?php

namespace App\Repositories\Eloquent;

use App\Enum\ListingStatus;
use App\Models\Listing;
use App\Repositories\Contracts\ListingRepositoryInterface;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Pagination\LengthAwarePaginator;

class ListingRepository extends BaseRepository implements ListingRepositoryInterface
{
    /**
     * Get featured listings.
     *
     * @param int $limit
     * @param array $with
     * @return Collection
     */
    public function getFeatured(int $limit = 12, array $with = ['specification', 'condition', 'owner']): Collection
    {
        return $this->model
            ->with($with)
            ->where('status', ListingStatus::APPROVED)
            ->latest()
            ->limit($limit)
            ->get();
    }

    /**
     * Get approved listings with pagination.
     *
     * @param int $perPage
     * @param array $with
     * @param array $filters
     * @return LengthAwarePaginator
     */
    public function getApprovedPaginated(
        int $perPage = 15,
        array $with = ['specification', 'condition', 'owner'],
        array $filters = []
    ): LengthAwarePaginator {
        $query = $this->model
            ->with($with)
            ->where('status', ListingStatus::APPROVED);

        // Apply filters
        if (!empty($filters['condition_id'])) {
            $query->where('condition_id', $filters['condition_id']);
        }

        if (!empty($filters['min_price'])) {
            $query->where('price', '>=', $filters['min_price']);
        }

        if (!empty($filters['max_price'])) {
            $query->where('price', '<=', $filters['max_price']);
        }

        if (!empty($filters['location'])) {
            $query->where('location', 'like', '%' . $filters['location'] . '%');
        }

        if (!empty($filters['search'])) {
            $query->where(function ($q) use ($filters) {
                $q->where('title', 'like', '%' . $filters['search'] . '%')
                    ->orWhere('description', 'like', '%' . $filters['search'] . '%');
            });
        }

        // Apply sorting
        $sortBy = $filters['sort_by'] ?? 'created_at';
        $sortOrder = $filters['sort_order'] ?? 'desc';
        $query->orderBy($sortBy, $sortOrder);

        return $query->paginate($perPage);
    }

    /**
     * Get listings by owner (dealer).
     *
     * @param int $ownerId
     * @param array $with
     * @param array $filters
     * @return LengthAwarePaginator
     */
    public function getByOwner(
        int $ownerId,
        array $with = ['specification', 'condition'],
        array $filters = []
    ): LengthAwarePaginator {
        $query = $this->model
            ->with($with)
            ->where('owner_id', $ownerId);

        // Apply status filter if provided
        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        $sortBy = $filters['sort_by'] ?? 'created_at';
        $sortOrder = $filters['sort_order'] ?? 'desc';
        $query->orderBy($sortBy, $sortOrder);

        $perPage = $filters['per_page'] ?? 15;
        return $query->paginate($perPage);
    }

    /**
     * Get related listings (same condition or similar price range).
     *
     * @param string $listingId
     * @param int $limit
     * @param array $with
     * @return Collection
     */
    public function getRelated(string $listingId, int $limit = 4, array $with = ['specification', 'condition', 'owner']): Collection
    {
        $listing = $this->findByUuid($listingId, ['id', 'condition_id', 'price']);

        if (!$listing) {
            return collect();
        }

        $query = $this->model
            ->with($with)
            ->where('status', ListingStatus::APPROVED)
            ->where('id', '!=', $listing->id);

        // Get listings with same condition
        if ($listing->condition_id) {
            $query->where('condition_id', $listing->condition_id);
        }

        // Also include listings in similar price range (±20%)
        if ($listing->price) {
            $priceRange = $listing->price * 0.2;
            $query->orWhereBetween('price', [
                $listing->price - $priceRange,
                $listing->price + $priceRange
            ]);
        }

        return $query
            ->limit($limit)
            ->inRandomOrder()
            ->get();
    }

    /**
     * Get listing by UUID with relations.
     *
     * @param string $uuid
     * @param array $with
     * @return Model|null
     */
    public function findByUuidWith(string $uuid, array $with = ['specification', 'condition', 'owner', 'features']): ?Model
    {
        return $this->model
            ->with($with)
            ->where('uuid', $uuid)
            ->first();
    }

    /**
     * Get listings by condition.
     *
     * @param int $conditionId
     * @param int $perPage
     * @param array $with
     * @return LengthAwarePaginator
     */
    public function getByCondition(
        int $conditionId,
        int $perPage = 15,
        array $with = ['specification', 'condition', 'owner']
    ): LengthAwarePaginator {
        return $this->model
            ->with($with)
            ->where('status', ListingStatus::APPROVED)
            ->where('condition_id', $conditionId)
            ->latest()
            ->paginate($perPage);
    }

    /**
     * Get listing statistics.
     *
     * @param int|null $ownerId
     * @return array
     */
    public function getStatistics(?int $ownerId = null): array
    {
        $query = $this->model->newQuery();

        if ($ownerId) {
            $query->where('owner_id', $ownerId);
        }

        return [
            'total' => $query->count(),
            'approved' => (clone $query)->where('status', ListingStatus::APPROVED)->count(),
            'pending' => (clone $query)->where('status', ListingStatus::PENDING)->count(),
            'rejected' => (clone $query)->where('status', ListingStatus::REJECTED)->count(),
        ];
    }
}

