<?php

namespace App\Services\ActivityLogs;

use App\Models\User;
use App\Services\ActivityLogs\ActivityLogBaseService;

class AuthLogService extends ActivityLogBaseService
{
    /**
     * Initialize the service
     *
     * @param \App\Models\User|null $user
     */
    public function __construct(
        User|null $user = null,
    ) {
        parent::__construct('auth', $user ?? auth()->user());
    }

    /**
     * Log user's login activity
     *
     * @return void
     */
    public function loggedIn()
    {
        $this->log(
            activity: 'logged_in',
            description: "User logged in with {$this->user->email}."
        );
    }

    /**
     * Log user's registration activity
     *
     * @return void
     */
    public function registered()
    {
        $this->log(
            activity: 'registered',
            description: "User registered with {$this->user->email}."
        );
    }

    /**
     * Log user's logout activity
     *
     * @return void
     */
    public function loggedOut()
    {
        $this->log(
            activity: 'logged_out',
            description: "User logged out with {$this->user->email}."
        );
    }

    /**
     * Log activity when email verification notification has been sent
     *
     * @return void
     */
    public function emailVerificationLinkRequested()
    {
        $this->log(
            activity: 'email_verification_link_requested',
            description: "Email verification notification has been sent to {$this->user->email}."
        );
    }

    /**
     * Log activity when user's email has been verified
     *
     * @return void
     */
    public function emailVerified(User $subjectUser = null)
    {
        $this->log(
            activity: 'email_verified',
            description: $subjectUser
                ? $subjectUser?->email . ' has been verified by ' . $this->user->email . '.'
                : "{$this->user->email} has been verified.",
        );
    }

    /**
     * Log password reset link request activity
     *
     * @return void
     */
    public function passwordResetLinkRequested()
    {
        $this->log(
            activity: 'password_reset_link_requested',
            description: "{$this->user->email} has requested password reset link."
        );
    }

    /**
     * Log password reset activity
     *
     * @return void
     */
    public function passwordReset()
    {
        $this->log(
            activity: 'password_reset',
            description: "{$this->user->email} has reset the password."
        );
    }
}
