<?php

namespace App\Traits;

use Illuminate\Support\Str;
use FFMpeg\Format\Video\X264;
use Illuminate\Support\Facades\Log;
use Intervention\Image\ImageManager;
use Illuminate\Support\Facades\Storage;
use ProtoneMedia\LaravelFFMpeg\Support\FFMpeg;

trait FileUploader
{
    protected function uploadPublicFile($inputFile, string $directory, string $oldFile = null): string
    {
        $this->makeDirectoryIfNotExists($directory);

        $newFilePath = Storage::disk('public')->putFileAs(
            $directory,
            $inputFile,
            $this->generateFileNameFromInput($inputFile)
        );

        $this->removePublicFile($oldFile);

        return $newFilePath;
    }

    protected function uploadPublicImage($inputFile, string $directory, string $oldImage = null): string
    {
        $this->makeDirectoryIfNotExists($directory);

        $fileName = $this->generateFileNameFromInput($inputFile);

        // Generate thumbnail
        $this->generateThumbnail($inputFile, $directory, $fileName, $oldImage);

        $newFilePath = Storage::disk('public')->putFileAs(
            $directory,
            $inputFile,
            $fileName
        );

        $this->removePublicFile($oldImage);

        return $newFilePath;
    }

    private function generateFileNameFromInput($inputFile, string $extension = null): string
    {
        $extension = !empty($extension) ? $extension : $inputFile->getClientOriginalExtension();
        return sprintf('%s_%s.%s', Str::uuid(), time(), $extension);
    }

    private function generateThumbnail($inputFile, string $directory, string $fileName, string $oldImage = null): void
    {
        $this->makeDirectoryIfNotExists("thumbnails/$directory");

        $img = ImageManager::imagick()->read($inputFile);

        $img->scale(285, null);

        $img->toPng();

        $img->save(storage_path("app/public/thumbnails/$directory/$fileName"));

        $this->removePublicFile('thumbnails/' . $oldImage);
    }

    private function removePublicFile(string $filePath = null): void
    {
        if (
            !empty($filePath)
            && Storage::disk('public')->exists($filePath)
        ) {
            Storage::disk('public')->delete($filePath);
        }
    }

    private function makeDirectoryIfNotExists(string $directory): void
    {
        if (!Storage::disk('public')->exists($directory)) {
            Storage::disk('public')->makeDirectory("$directory");
        }
    }

    protected function uploadPublicVideo($inputFile, string $directory, string $oldFile = null, $isShortVideo = false): array
    {
        $this->makeDirectoryIfNotExists($directory);

        $fileName = $this->generateFileNameFromInput($inputFile);

        $data = $this->compressAndUploadVideo($inputFile, $directory, $oldFile, $fileName, $isShortVideo);

        $this->removePublicFile($oldFile);

        return $data;
    }

    protected function compressAndUploadVideo($file, string $directory, ?string $oldFile, $fileName, $isShortVideo = false): array
    {
        $this->removePublicFile($oldFile);

        // Create temporary local file to avoid frequent S3 operations
        $tempFile = $this->prepareLocalTempFile($file);

        $compressedFileName = 'compressed_' . $fileName;
        $compressedPath = "{$directory}/videos/$compressedFileName";

        try {
            // compress short video
            $shortVideo = $isShortVideo ? $this->generateShortVideo($file, $directory, $tempFile, $oldFile, $fileName) : null;

            // Open video correctly using ProtoneMedia Laravel-FFMpeg
            $video = FFMpeg::fromDisk('local')->open($tempFile);
            $durationInSeconds = $video->getFormat()->get('duration'); // Get duration in seconds

            // Compress video locally first
            $video->export()
                ->toDisk('public')
                ->inFormat(new X264('aac', 'libx264'))
                ->save($compressedPath);

            // After compression, upload to S3
            return [
                'mime_type' => mime_content_type(storage_path("app/public/$compressedPath")),
                'filesize' => filesize(storage_path("app/public/$compressedPath")),
                'thumbnail' => $this->generateThumbnailFromCompressedVideo($directory, $compressedPath),
                'long_video' => $compressedPath,
                'long_video_time_duration' => round($durationInSeconds, 2),
                'short_video' => $shortVideo['short_video'] ?? null,
                'short_video_time_duration' => $shortVideo['short_video_time_duration'] ?? null,
            ];
        } catch (\Exception $e) {
            Log::error($e->getMessage(), $e->getTrace());
            throw new \RuntimeException('Video compression failed.');
        } finally {
            if (Storage::disk('local')->exists($tempFile)) {
                Storage::disk('local')->delete($tempFile);
            }
        }
    }

    public function generateShortVideo($file, string $directory, $tempFile, ?string $oldFile, $fileName)
    {
        $this->removePublicFile($oldFile);

        $compressedFileName = 'compressed_' . $fileName;
        $shortVideoPath = "{$directory}/short_videos/{$compressedFileName}";

        try {
            // Open video correctly using ProtoneMedia Laravel-FFMpeg
            $video = FFMpeg::fromDisk('local')->open($tempFile);
            // $durationInSeconds = $video->getFormat()->get('duration'); // Get duration in seconds
            $durationInSeconds = 20;

            // compress short video
            $video->export()
                ->toDisk('public')
                ->inFormat(new X264('aac', 'libx264'))
                ->addFilter(['-t', $durationInSeconds])
                ->save($shortVideoPath);

            return [
                'short_video' => $shortVideoPath,
                'short_video_time_duration' => round($durationInSeconds, 2),
            ];
        } catch (\Exception $e) {
            Log::error($e->getMessage(), $e->getTrace());
            throw new \RuntimeException('Short Video compression failed.');
        }
    }

    public function generateThumbnailFromCompressedVideo($directory, $compressedVideoPath)
    {
        try {
            $thumbnailFileName = Str::random() . '_thumbnail.jpg';
            $thumbnailPath = "$directory/thumbnails/{$thumbnailFileName}";

            if (!file_exists(storage_path("app/public/{$compressedVideoPath}"))) throw new \RuntimeException('Compressed video file not found.');

            // Generate the thumbnail from the compressed video at 1 second mark
            FFMpeg::fromDisk('public')
                ->open($compressedVideoPath)
                ->getFrameFromSeconds(1)
                ->export()
                ->toDisk('public')
                ->save($thumbnailPath);

            return $thumbnailPath;
        } catch (\Exception $exception) {
            Log::error('Thumbnail generation failed: ' . $exception->getMessage(), [
                'compressed_video_path' => $compressedVideoPath,
            ]);
            throw new \RuntimeException($exception->getMessage());
        }
    }

    protected function prepareLocalTempFile($file)
    {
        $tempDir = 'temp';
        $this->makeDirectoryIfNotExists($tempDir);
        $extension = $file instanceof \Illuminate\Http\UploadedFile
            ? $file->getClientOriginalExtension()
            : pathinfo($file->getPathname(), PATHINFO_EXTENSION);

        $tempFileName = Str::random(32) . '.' . ($extension ?: 'bin');

        $tempFilePath = $tempDir . '/' . $tempFileName;
        $file->move(storage_path("app/private/temp"), $tempFileName);

        return $tempFilePath;
    }
}
