<?php

namespace App\Traits;

use Exception;
use App\Models\UserDevice;
use Illuminate\Support\Str;
use Kreait\Firebase\Factory;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use App\Notifications\PushNotificationManager;
use Kreait\Firebase\Messaging\RawMessageFromArray;

trait PushNotification
{
    public static function sendNotificationsToUsers($users, $notification)
    {
        try {
            foreach ($users as $user) {
                $tokens = $user->devices->pluck('fcm_token')->unique()->toArray();
                if (!empty($tokens)) {
                    self::sendFirebaseMultiDeviceCloudMessage($tokens, $notification);
                }
            }
        } catch (Exception $e) {
            Log::info($e->getMessage());
        }
    }

    public static function sendFirebaseMultiDeviceCloudMessage($devices, $notification)
    {
        try {
            if (!empty($devices)) {
                $firebasePath = base_path('iduniq-23a71-firebase-adminsdk-fbsvc-128fbce7c5.json');
                $firebase = (new Factory)->withServiceAccount($firebasePath);
                $messaging = $firebase->createMessaging();

                $devicesChunks = array_chunk($devices, 90);

                foreach ($devicesChunks as $deviceChunk) {
                    $message = [
                        'notification' => [
                            'title' => $notification['title'],
                            'body' => $notification['body'],
                        ],
                        'data' => $notification['payload'],
                        'webpush' => [
                            'notification' => [
                                'title' => $notification['title'],
                                'body' => $notification['body'],
                            ],
                        ],
                        'fcm_options' => [
                            'analytics_label' => 'some-analytics-label'
                        ]
                    ];

                    $result = $messaging->sendMulticast(new RawMessageFromArray($message), $deviceChunk);

                    if ($result->hasFailures()) {
                        foreach ($result->failures()->getItems() as $failure) {
                            $errorMsg = $failure->error()->getMessage();
                            Log::error("FCM failure: " . $errorMsg);
                            if (Str::contains($errorMsg, 'entity was not found')) {
                                UserDevice::whereIn('fcm_token', $deviceChunk)->delete();
                            } else {
                                Log::error($failure->error()->getMessage() . PHP_EOL);
                            }
                        }
                    }
                }
            }

            return null;
        } catch (\Exception $e) {
            Log::error($e->getMessage(), $e->getTrace());
        }
    }

    public static function sendNotification($users, $title, $body, array $payload = [])
    {
        $notification =  [
            'title'         => $title,
            'body'          => $body,
            'payload'       => $payload,
        ];

        Notification::send($users, new PushNotificationManager($notification, $users));
    }

    public static function storeUserDevice($deviceToken, $accessToken = null)
    {
        try {
            if (!$accessToken) {
                Log::error("No authenticated user and no access token provided.");
                return;
            }

            $tokenId = strpos($accessToken, '|') !== false
                ? explode('|', $accessToken)[0]
                : auth()->user()->currentAccessToken()?->id;

            if (!$tokenId) {
                Log::error("Invalid access token format.");
                return;
            }

            $token = \Laravel\Sanctum\PersonalAccessToken::find($tokenId);

            if (!$token) {
                Log::error("Invalid or expired token.");
                return;
            }

            UserDevice::updateOrCreate([
                'user_id' => $token->tokenable_id,
                'personal_access_token_id' => $token->id,
            ], [
                'fcm_token' => $deviceToken,
            ]);
        } catch (\Exception $exception) {
            Log::error("Error saving device information: " . $exception->getMessage());
        }
    }

    public static function removeUserDevice($token)
    {
        try {
            UserDevice::where('personal_access_token_id', $token->id)->delete();
        } catch (\Exception $exception) {
            Log::error("Error removing device information: " . $exception->getMessage());
        }
    }
}
